#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/poll.h>
#include <unistd.h>
#include <signal.h>
#include <rpc/rpc.h>
#include <errno.h>


#define MOUNT_PROGRAM   100005
#define MOUNT_VERSION   1

#define NFS_PROGRAM     100003
#define NFS_VERSION     2

#define XACT_CACHE_SIZE 20

struct svcudp_data {
    u_int su_iosz;              /* byte size of send.recv buffer */
    u_long su_xid;              /* transaction id */
    XDR su_xdrs;                /* XDR handle */
    char su_verfbody[MAX_AUTH_BYTES];   /* verifier body */
    char *su_cache;             /* cached data, NULL if no cache */
};

typedef struct rpc_act_entry_s {
    unsigned long xid;
    unsigned long prog;
    unsigned long vers;
} rpc_act_entry_t;

static int rpc_fd;
static SVCXPRT *transp_svc;
static int shutdown_pipe[2];
static int shutdown_signalled=0;
static rpc_act_entry_t last_entries[XACT_CACHE_SIZE];
static int current_entry_idx=0;

extern void nfs_program_2(struct svc_req *rqstp, register SVCXPRT *transp);
extern void mountprog_1(struct svc_req *rqstp, register SVCXPRT *transp);


static int
rpc_xact_check_valid (rpc_act_entry_t *new_ent)
{
    int i;

    for (i = 0; i < XACT_CACHE_SIZE; i++) {
        if (new_ent->xid == last_entries[i].xid &&
            new_ent->prog == last_entries[i].prog &&
            new_ent->vers == last_entries[i].vers) {
            return 0;
        }
    }

    last_entries[current_entry_idx] = *new_ent;
    current_entry_idx = (current_entry_idx + 1) % XACT_CACHE_SIZE;

    return 1;
}


static void
sig_handler (int signum)
{
    char dummy;

    if (!shutdown_signalled) {
        shutdown_signalled = 1;
        write (shutdown_pipe[1], &dummy, 1);
    }
}


static void
dispatch_rpc (struct svc_req *r, SVCXPRT *xprt)
{
    rpc_act_entry_t ent;

    ent.xid = ((struct svcudp_data *)xprt->xp_p2)->su_xid;
    ent.prog = r->rq_prog;
    ent.vers = r->rq_vers;

    if (rpc_xact_check_valid (&ent) == 0) {
            fprintf (stderr, "Found duplicate entry: %lu\n", ent.xid);
        return;
    }

    /* Handle NFS RPC requests */
    switch (r->rq_prog) {

    case NFS_PROGRAM:
        if (r->rq_vers != NFS_VERSION) {
            break;
        }
        /*
        fprintf (stderr, "NFS RPC request found (proc=%ld, xid=%lu)\n", 
                 r->rq_proc, ent.xid);
        */
        nfs_program_2 (r, xprt);

        return;

    case MOUNT_PROGRAM:
        if (r->rq_vers != MOUNT_VERSION) {
            break;
        }
        /*
        fprintf (stderr, "Mount RPC request found (proc=%ld, xid=%lu)\n", 
                 r->rq_proc, ent.xid);
        */
        mountprog_1 (r, xprt);

        return;
    }

    fprintf (stderr, "%s: error rpc prog: %ld, vers: %ld not supported\n",
             __FUNCTION__, r->rq_prog, r->rq_vers);

    return;
}


static int
rpc_startup (unsigned short port)
{
    int rc;
    struct sockaddr_in uaddr;
    unsigned long prognum, version;

    memset (&last_entries, 0, 20 * sizeof (rpc_act_entry_t));

    uaddr.sin_family = AF_INET;
    uaddr.sin_addr.s_addr = 0;
    uaddr.sin_port = htons(port);
  
    rpc_fd = socket (AF_INET, SOCK_DGRAM, 0);
    if (rpc_fd < 0) {
        fprintf (stderr, "%s: cannot create dgram socket (errno=%d)\n", 
                 __FUNCTION__, errno);
        return errno;
    }

    rc = bind (rpc_fd, (struct sockaddr*)&uaddr, sizeof(uaddr));
    if (rc < 0) {
        fprintf (stderr, "%s: cannot bind to port %d (errno=%d)\n", 
                 __FUNCTION__, port, errno);
        close (rpc_fd);
        return errno;
    }
  
    transp_svc = svcudp_create(rpc_fd);
    if (transp_svc == NULL) {
        fprintf (stderr, "%s: cannot register socket to svcudp_create\n", 
                 __FUNCTION__);
        close (rpc_fd);
        return 1;
    }

    prognum = NFS_PROGRAM;
    version = NFS_VERSION;
    svc_unregister(prognum, version);    
    if (!svc_register(transp_svc, prognum, version, dispatch_rpc, 0)) {
        fprintf (stderr, "%s: cannot register prognum=%ld, version=%ld\n",
                 __FUNCTION__, prognum, version);
        return 1;
    }

    prognum = MOUNT_PROGRAM;
    version = MOUNT_VERSION;
    svc_unregister(prognum, version);    
    if (!svc_register(transp_svc, prognum, version, dispatch_rpc, 0)) {
        fprintf (stderr, "%s: cannot register prognum=%ld, version=%ld\n",
                 __FUNCTION__, prognum, version);
        return 1;
    }

    return 0;
}


static void
rpc_shutdown (void)
{
    close (rpc_fd);
}


static int
dispatch_init (void)
{
    if (pipe (shutdown_pipe) < 0) {
        fprintf (stderr, "%s: error setting up shutdown pipe (errno=%d)\n",
                 __FUNCTION__, errno);
        return errno;
    } 

    if (signal (SIGINT, sig_handler) == SIG_ERR) {
        fprintf (stderr, "%s: error installing signal handler (errno=%d)\n",
                 __FUNCTION__, errno);
        return errno;
    }

    return 0;
}


static void
dispatch_shutdown (void)
{
    close (shutdown_pipe[0]);
    close (shutdown_pipe[1]);
}


static void
handle_rpc (short revents)
{
    if (revents & POLLIN) {
        //    svc_getreq_dispatch (rpc_fd, dispatch_rpc);
        svc_getreq_common (rpc_fd);
    } else {
        fprintf (stderr, "%s: unknown event on rpc_fd %#x, shutting down\n",
                 __FUNCTION__, revents);
        shutdown_signalled = 1;
    }
}


static void
handle_shutdown (short revents)
{
    char dummy;

    if (revents & POLLIN) {
        read (shutdown_pipe[0], &dummy, 1);
    } else {
        fprintf (stderr, "%s: unknown event on pipe %#x, shutting down\n",
                 __FUNCTION__, revents);
        shutdown_signalled = 1;
    }
}


static void
dispatch_loop (void)
{
    struct pollfd poll_array[2];
    int rc;

    poll_array[0].fd = rpc_fd;
    poll_array[0].events = POLLIN;

    poll_array[1].fd = shutdown_pipe[0];
    poll_array[1].events = POLLIN;

    while (!shutdown_signalled) {
        rc = poll (poll_array, 2, -1);

        if (rc < 0) {
            if (errno == EINTR) {
                continue;
            }
            fprintf (stderr, "%s: poll error (errno=%d)\n", 
                     __FUNCTION__, errno);
            return;
        }

        if (poll_array[0].revents) {
            handle_rpc (poll_array[0].revents);
        }
    
        if (poll_array[1].revents) {
            handle_shutdown (poll_array[1].revents);
        }
    }
}


static void
usage (char *prog_name)
{
    fprintf (stderr, "Usage: %s -p (port_num)\n", prog_name);
}


int
main (int argc, char **argv)
{
    char c;
    unsigned short port=0;
    int rc;

    while ((c = getopt (argc, argv, "p:")) != -1) {
        switch (c) {
        case 'p':
            port = atoi(optarg);
            break;
        default:
            usage (argv[0]);
            exit (1);
        }
    }
  
    if (port == 0) {
        usage (argv[0]);
        exit (1);
    }

    rc = rpc_startup (port);
    if (rc) {
        exit (rc);
    }
  
    rc = dispatch_init ();
    if (rc) {
        exit (rc);
    }

    dispatch_loop ();

    rpc_shutdown ();
    dispatch_shutdown ();

    return 0;
}

/*
 * Local variables:
 *  c-basic-offset: 4
 *  indent-tabs-mode: nil
 * End:
 */
